/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import com.inet.report.*;

/**
 * This sample shows how to create a new report and add some database fields from a stored procedure to it. The Data
 * Source Configuration "Sample Database" is necessary for this sample. You can create it using Data Source Manager in
 * i-net Designer or the Configuration Manager.
 */
public class AddDatabaseFieldsSample extends RDCSample {

    /**
     * Creates a new report and add some database fields from a stored procedure to it.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {

        Engine eng = null;
        try {
            // Create an Engine for a new report
            eng = RDC.createEmptyEngine( exportFmt );

            // Fill the engine
            final DatabaseTables dbTables = eng.getDatabaseTables();
            final Fields fields = eng.getFields();

            // Define a data source
            Datasource ds = dbTables.createDatasource( "Sample Database" ); // Data Source Configuration "Sample Database"

            // Define the stored procedure for the data source
            TableSource ts_Employees = ds.createTableSource( "Employees" );

            String[] columnNames = { "EmployeeID", "FirstName", "LastName" };
            int[] columnTypes = { Field.NUMBER, Field.STRING, Field.STRING };
            for( int colIdx = 0; colIdx < columnNames.length; colIdx++ ) {
                ts_Employees.addColumn( columnNames[colIdx], columnTypes[colIdx] );
            }

            // Add database fields and text fields to the report
            DatabaseField dbField = null;
            Area dArea = eng.getArea( "D" );
            Section dSec = dArea.getSection( 0 );
            Area phArea = eng.getArea( "PH" );
            Section phSec = phArea.getSection( 0 );

            dbField = fields.getDatabaseField( ts_Employees.getAlias() + ".EmployeeID" );
            dSec.addFieldElement( dbField, 700, 100, 1000, 200 );
            Text text = phSec.addText( 700, 100, 2000, 500 );
            Paragraph para = text.addParagraph();
            TextPart tPart = para.addTextPart( dbField.getShortName() );
            tPart.setBold( true );
            tPart.setUnderline( true );

            dbField = fields.getDatabaseField( ts_Employees.getAlias() + ".FirstName" );
            dSec.addFieldElement( dbField, 4500, 100, 2000, 200 );
            text = phSec.addText( 4500, 100, 2000, 500 );
            para = text.addParagraph();
            tPart = para.addTextPart( dbField.getShortName() );
            tPart.setBold( true );
            tPart.setUnderline( true );

            dbField = fields.getDatabaseField( ts_Employees.getAlias() + ".LastName" );
            dSec.addFieldElement( dbField, 7800, 100, 2000, 200 );
            text = phSec.addText( 7800, 100, 2000, 500 );
            para = text.addParagraph();
            tPart = para.addTextPart( dbField.getShortName() );
            tPart.setBold( true );
            tPart.setUnderline( true );

        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
        }
        return eng;
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new AddDatabaseFieldsSample().initUI();
    }
}
